<?php

namespace common\models;

use Yii;
use common\helpers\Date;

/**
 * This is the model class for table "plan".
 *
 * @property integer $id
 * @property integer $source_id
 * @property integer $vote_id
 * @property integer $category_id
 * @property integer $quarter_id
 * @property integer $para_year
 * @property string $item_no
 * @property string $title
 * @property integer $timeline
 * @property string $responsible_personal
 * @property string $status
 * @property integer $reporting_detail_id
 * @property integer $current_previous
 * @property integer $length_term
 * @property integer $created_at
 * @property integer $created_by
 * @property integer $updated_at
 * @property integer $updated_by
 *
 * @property AuditComment[] $auditComments
 * @property Implementation[] $implementations
 * @property ManagementResponse[] $managementResponses
 * @property Observation[] $observations
 * @property Category $category
 * @property FinancialYear $paraYear
 * @property Quarter $quarter
 * @property ReportingDetail $reportingDetail
 * @property Source $source
 * @property Vote $vote
 * @property Recommendation[] $recommendations
 * @property RemedialAction[] $remedialActions
 * @property StatusAudit[] $statusAudits
 */
class Plan extends \yii\db\ActiveRecord
{

    const STATUS_NOT_IMPLEMENTED_VALUE = 0;
    const STATUS_INPROGRESS_VALUE = 1;
    const STATUS_IMPLEMENTED_VALUE = 3;
    const STATUS_OVERTAKEN_VALUE = 4;
    const STATUS_WAITING_CAG_VALUE = 2;
    const TERM_SHORT = 1;
    const TERM_MEDIUM = 2;
    const TERM_LONG = 3;
    const TERM_SHORT_NAME = 'SHORT';
    const TERM_MEDIUM_NAME = 'MEDIUM';
    const TERM_LONG_NAME = 'LONG';
    const TERM_SHORT_MONTH = 6;
    const TERM_MEDIUM_MONTH = 12;
    const TERM_LONG_MONTH = 13;
    const STATUS_NOT_IMPLEMENTED_NAME = 'Not attended';
    const STATUS_INPROGRESS_NAME = 'In Progress';
    const STATUS_IMPLEMENTED_NAME = 'Implemented & Closed';
    const STATUS_OVERTAKEN_NAME = 'Overtaken by events & Closed';
    const STATUS_WAITING_CAG_NAME = 'implemented But Waiting for CAG';

    public $evidence;
    public $deadline;
    public $priority_id;
    const PRIORITY_HIGH = 1;
    const PRIORITY_MEDIUM = 2;
    const PRIORITY_HIGH_NAME = 'High';
    const PRIORITY_LOW_NAME = 'Low';
    const PRIORITY_MEDIUM_NAME = 'Medium';

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'plan';
    }

    public function behaviors()
    {
        return [
            'bedezign\yii2\audit\AuditTrailBehavior'
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['source_id', 'vote_id', 'priority_id', 'category_id', 'para_year', 'item_no', 'title', 'reporting_detail_id'], 'required'],
            [['assigned_user_id', 'risk', 'source_id', 'vote_id', 'last_reviewed_at', 'last_reviewed_by', 'category_id', 'quarter_id', 'para_year', 'timeline', 'reporting_detail_id', 'current_previous', 'length_term', 'created_at', 'created_by', 'updated_at', 'updated_by'], 'integer'],
            [['item_no'], 'string', 'max' => 100],
            [['title', 'deadline', 'responsible_personal', 'evidence', 'risk'], 'string', 'max' => 255],
            [['status'], 'string', 'max' => 45],
            [['item_no'], 'validatePara'],
            ['item_no', 'match', 'pattern' => '/^([0-9]\s?)+([.]\s?([0-9]\s?)+)*$/', 'message' => 'Invalid format, use format e.g 4.5.1'],
            [['category_id'], 'exist', 'skipOnError' => true, 'targetClass' => Category::className(), 'targetAttribute' => ['category_id' => 'id']],
            [['para_year'], 'exist', 'skipOnError' => true, 'targetClass' => FinancialYear::className(), 'targetAttribute' => ['para_year' => 'id']],
            [['quarter_id'], 'exist', 'skipOnError' => true, 'targetClass' => Quarter::className(), 'targetAttribute' => ['quarter_id' => 'id']],
            [['reporting_detail_id'], 'exist', 'skipOnError' => true, 'targetClass' => ReportingDetail::className(), 'targetAttribute' => ['reporting_detail_id' => 'id']],
            [['source_id'], 'exist', 'skipOnError' => true, 'targetClass' => Source::className(), 'targetAttribute' => ['source_id' => 'id']],
            [['vote_id'], 'exist', 'skipOnError' => true, 'targetClass' => Vote::className(), 'targetAttribute' => ['vote_id' => 'id']],
        ];
    }

    public function validatePara($attribute, $params, $validator)
    {
        if ($this->quarter_id)
            $exists = Plan::find()
                ->where(['para_year' => $this->para_year, 'vote_id' => $this->vote_id, 'item_no' => $this->item_no, 'quarter_id' => $this->quarter_id])
                ->one();
        else
            $exists = Plan::find()
                ->where(['para_year' => $this->para_year, 'vote_id' => $this->vote_id, 'item_no' => $this->item_no])
                ->one();
        if ($exists) {
            $this->addError($attribute, 'This finding /Directive  already exists');
        }
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'assigned_user_id'=>'Assignee',
            'source_id' => 'Source',
            'vote_id' => 'Vote',
            'group_id' => 'Group',
            'risk' => 'Risk',
            'category_id' => 'Category',
            'quarter_id' => 'Quarter',
            'para_year' => 'Year of finding',
            'item_no' => 'Para No.',
            'priority_id' => 'Priority',
            'title' => 'Finding / Directive title',
            'evidence_implemented' => 'Enter Working details,reference or file no.',
            'evidence' => 'Enter Working details,reference or file no.',
            'timeline' => 'Deadline date',
            'responsible_personal' => 'Responsible Personal',
            'status' => 'Status',
            'reporting_detail_id' => 'Reporting Detail ID',
            'current_previous' => '0 =current issue
1 =previous issue',
            'length_term' => '1 = short_term
2 =medium_term
3 = long_term',
            'created_at' => 'Created At',
            'created_by' => 'Created By',
            'updated_at' => 'Updated At',
            'updated_by' => 'Updated By',
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAssignedUser()
    {
        return $this->hasOne(User::className(), ['id' => 'assigned_user_id']);
    }
    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuditComments()
    {
        return $this->hasMany(AuditComment::className(), ['plan_id' => 'id']);
    }
    public function getAuditCommittee()
    {
        return $this->hasMany(AuditCommittee::className(), ['plan_id' => 'id']);
    }
    public function getBoardDirective()
    {
        return $this->hasMany(BoardDirective::className(), ['plan_id' => 'id']);
    }


    /**
     * @return \yii\db\ActiveQuery
     */
    public function getImplementations()
    {
        //    return $this->hasMany(Implementation::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getManagementResponses()
    {
        return $this->hasMany(ManagementResponse::className(), ['plan_id' => 'id']);
    }

    public function getIssueReviews()
    {
        return $this->hasMany(IssueReview::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getObservations()
    {
        return $this->hasMany(Observation::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getCategory()
    {
        return $this->hasOne(Category::className(), ['id' => 'category_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getParaYear()
    {
        return $this->hasOne(FinancialYear::className(), ['id' => 'para_year']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getQuarter()
    {
        return $this->hasOne(Quarter::className(), ['id' => 'quarter_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getReportingDetail()
    {
        return $this->hasOne(ReportingDetail::className(), ['id' => 'reporting_detail_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSource()
    {
        return $this->hasOne(Source::className(), ['id' => 'source_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getVote()
    {
        return $this->hasOne(Vote::className(), ['id' => 'vote_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRecommendations()
    {
        return $this->hasMany(Recommendation::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getActionsTaken()
    {
        return $this->hasMany(ActionTaken::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getRemedialActions()
    {
        return $this->hasMany(RemedialAction::className(), ['plan_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getStatusAudits()
    {
        // return $this->hasMany(StatusAudit::className(), ['plan_id' => 'id']);
    }

    public static function getPriorityId($name)
    {
        switch ($name) {
            case self::PRIORITY_HIGH_NAME:
                return '<span style="font-weight:bold; color:red">' . self::PRIORITY_HIGH_NAME . '</span>';

                break;
            case self::PRIORITY_MEDIUM_NAME:
                return '<span style="font-weight:bold; color:green">' . self::PRIORITY_MEDIUM_NAME . '</span>';
                break;
            default:
                return "-";
                break;
        }
    }

    public function isAssigned()
    {
        return Yii::$app->user->id == $this->assigned_user_id;
    }

    public static function getStatusById($id)
    {
        switch ($id) {
            case self::STATUS_NOT_IMPLEMENTED_VALUE:
                return '<span style="font-weight:bold; color:black">' . self::STATUS_NOT_IMPLEMENTED_NAME . '</span>';

                break;
            case self::STATUS_IMPLEMENTED_VALUE:
                return '<span style="font-weight:bold; color:green">' . self::STATUS_IMPLEMENTED_NAME . '</span>';
                break;
            case self::STATUS_OVERTAKEN_VALUE:
                return '<span style="font-weight:bold; color:blue">' . self::STATUS_OVERTAKEN_NAME . '</span>';

                break;

            case self::STATUS_INPROGRESS_VALUE:
                return '<span style="font-weight:bold; color:blue">' . self::STATUS_INPROGRESS_NAME . '</span>';

                break;
            case self::STATUS_WAITING_CAG_VALUE:
                return '<span style="font-weight:bold; color:red">' . self::STATUS_WAITING_CAG_NAME . '</span>';

                break;


            default:
                return "-";
                break;
        }
    }

    public function getStatusname()
    {
        switch ($this->status) {
            case self::STATUS_NOT_IMPLEMENTED_VALUE:
                return '<span style="font-weight:bold; color:black">' . self::STATUS_NOT_IMPLEMENTED_NAME . '</span>';

                break;
            case self::STATUS_IMPLEMENTED_VALUE:
                return '<span style="font-weight:bold; color:green">' . self::STATUS_IMPLEMENTED_NAME . '</span>';
                break;
            case self::STATUS_OVERTAKEN_VALUE:
                return '<span style="font-weight:bold; color:blue">' . self::STATUS_OVERTAKEN_NAME . '</span>';

                break;

            case self::STATUS_INPROGRESS_VALUE:
                return '<span style="font-weight:bold; color:grey">' . self::STATUS_INPROGRESS_NAME . '</span>';

                break;
            case self::STATUS_WAITING_CAG_VALUE:
                return '<span style="font-weight:bold; color:red">' . self::STATUS_WAITING_CAG_NAME . '</span>';

                break;


            default:
                return "-";
                break;
        }
    }

    public function getTermlength()
    {
        switch ($this->length_term) {
            case self::TERM_SHORT:
                return self::TERM_SHORT_NAME;

                break;
            case self::TERM_MEDIUM:
                return self::TERM_MEDIUM_NAME;

                break;
            case self::TERM_LONG:
                return self::TERM_LONG_NAME;

                break;


            default:
                return "-";
                break;
        }
    }

    /**
     * Determines term length for this issue
     * *** */
    public static function findTermLength($deadline, $initial_timestamp = null)
    {

        if (!$initial_timestamp) {
            $stringdate = date('Y') . '-04-01';
            $initial_timestamp = strtotime($stringdate . '00:00:00');
        }
        $shortdeadline = (self::TERM_SHORT_MONTH * 30 * 24 * 3600) + $initial_timestamp;
        $mediumdeadline = (self::TERM_MEDIUM_MONTH * 30 * 24 * 3600) + $initial_timestamp;;
        $longdeadline = (self::TERM_LONG_MONTH * 30 * 24 * 3600) + $initial_timestamp;
        if ($deadline <= $shortdeadline)
            return self::TERM_SHORT;

        elseif ($deadline <= $mediumdeadline)
            return self::TERM_MEDIUM;

        elseif ($deadline > $mediumdeadline)
            return self::TERM_LONG;
        else
            return "";
    }

    public static function statusoptions()
    {
        $options = [];
        $options[] = ['id' => self::STATUS_NOT_IMPLEMENTED_VALUE, 'name' => self::STATUS_NOT_IMPLEMENTED_NAME];
        $options[] = ['id' => self::STATUS_INPROGRESS_VALUE, 'name' => self::STATUS_INPROGRESS_NAME];
        $options[] = ['id' => self::STATUS_OVERTAKEN_VALUE, 'name' => self::STATUS_OVERTAKEN_NAME];
        $options[] = ['id' => self::STATUS_WAITING_CAG_VALUE, 'name' => self::STATUS_WAITING_CAG_NAME];
        $options[] = ['id' => self::STATUS_IMPLEMENTED_VALUE, 'name' => self::STATUS_IMPLEMENTED_NAME];
        return $options;
    }

    public function gettermoptions()
    {
        $options = [];
        $options[] = ['id' => self::TERM_SHORT, 'name' => self::TERM_SHORT_NAME];
        $options[] = ['id' => self::TERM_MEDIUM, 'name' => self::TERM_MEDIUM_NAME];
        $options[] = ['id' => self::TERM_LONG, 'name' => self::TERM_LONG_NAME];
        return $options;
    }
    public static function getPriorityOptions()
    {
        $options = [];
        $options[] = ['id' => self::PRIORITY_LOW_NAME, 'name' => self::PRIORITY_LOW_NAME];
        $options[] = ['id' => self::PRIORITY_MEDIUM_NAME, 'name' => self::PRIORITY_MEDIUM_NAME];
        $options[] = ['id' => self::PRIORITY_HIGH_NAME, 'name' => self::PRIORITY_HIGH_NAME];
        return $options;
    }
    public function getstatusoptions()
    {
        $options = [];
        $options[] = ['id' => self::STATUS_NOT_IMPLEMENTED_VALUE, 'name' => self::STATUS_NOT_IMPLEMENTED_NAME];
        $options[] = ['id' => self::STATUS_INPROGRESS_VALUE, 'name' => self::STATUS_INPROGRESS_NAME];
        $options[] = ['id' => self::STATUS_OVERTAKEN_VALUE, 'name' => self::STATUS_OVERTAKEN_NAME];
        if ($this->source_id == 2)
            $options[] = ['id' => self::STATUS_WAITING_CAG_VALUE, 'name' => self::STATUS_WAITING_CAG_NAME];
        $options[] = ['id' => self::STATUS_IMPLEMENTED_VALUE, 'name' => self::STATUS_IMPLEMENTED_NAME];
        return $options;
    }

    public static function detailoption($type)
    {
        switch ($type) {
            case 0:
                return "Observation";

                break;
            case 1:
                return "Recommendation";
                break;
            case 2:
                return "Audit Comment";

                break;
            case 7:
                return "Audit Committee Comment";

                break;
                //case 8:
                //    return "Board Directive";

                //    break;
            case 3:

                return "Management Response";
                break;
            case 4:
                return "Action To be Taken";

                break;
            case 5:
                return "Action Taken";

                break;
            case 6:
                return "Review";

                break;

            default:
                return "";
                break;
        }
    }

    /**
     * @
     * ******** */
    public function afterFind()
    {
        $this->deadline = Date::toDateString($this->timeline);
        return parent::afterFind();
    }

    public static function updateToPrevious($source_id, $vote_id)
    {
        return Plan::updateAll(['current_previous' => 1], 'source_id =' . $source_id . ' AND vote_id=' . $vote_id . ' AND status=' . self::STATUS_IMPLEMENTED_VALUE);
    }

    public function beforeSave($insert)
    {
        $model_quarter_id = $this->quarter_id;
        $model_year_id = $this->para_year;
        $current_year = \common\components\Globals::currentYear();
        $current_quarter = \common\components\Globals::currentQuarter();
        $current_year_id = $current_year->id;
        $current_quarter_id = $current_quarter->id;
        //annual reporting issues
        if ($model_year_id == $current_year_id) {
            //current issues
            $this->current_previous = 0;
            if ($model_quarter_id && $model_quarter_id == $current_quarter_id) {
                $this->current_previous = 1;
            }
            //set previous 
        } else {
            $this->current_previous = 1;
        }
        return parent::beforeSave($insert);
    }

    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        //set if previous year
        /*         * 8 $model_quarter_id = $this->quarter_id;
          $model_year_id = $this->para_year;
          $current_year_id = $current_year->id;
          $current_quarter_id = $current_quarter->id;
          //annual reporting issues
          if($model_year_id==$current_year_id){
          //current issues
          $this->current_previous=0;
          if($model_quarter_id && $model_quarter_id==$current_quarter_id){
          $this->current_previous=1;}
          //set previous
          } else{ $this->current_previous=1;}
          $this->save(false);
         * * */ /*         * * previous year setting ends ** */
        $vote = $this->vote;
        $vote->last_attended_at = time();
        $vote->last_attended_by = Yii::$app->user->id;

        $vote->update(false);
    }

    public static function setPreviousIssues()
    {
        $models = Plan::find()->all();
        $current_year = \common\components\Globals::currentYear();
        $current_quarter = \common\components\Globals::currentQuarter();
        foreach ($models as $model) {
            self::set_one_previous($model, $current_year, $current_quarter);
        }
    }

    public static function set_one_previous($model, $current_year, $current_quarter)
    {
        $model_quarter_id = $model->quarter_id;
        $model_year_id = $model->para_year;
        $current_year_id = $current_year->id;
        $current_quarter_id = $current_quarter->id;
        //annual reporting issues
        if ($model_year_id == $current_year_id) {
            //current issues
            $model->current_previous = 0;
            if ($model_quarter_id && $model_quarter_id == $current_quarter_id) {
                $model->current_previous = 1;
            }
            //set previous 
        } else {
            $model->current_previous = 1;
        }

        $model->save(false);
    }
}
