<?php

namespace common\models;

use common\helpers\Date;
use Yii;
use backend\models\UserForm;
use common\models\User;

/**
 * This is the model class for table "auditor".
 *
 * @property integer $id
 * @property integer $vote_id
 * @property integer $user_id
 * @property string $first_name
 * @property string $middle_name
 * @property string $last_name
 * @property string $title_position
 * @property string $phone_number
 * @property string $email
 * @property integer $gender
 * @property integer $data_of_birth
 * @property string $check_no
 * @property integer $date_of_appointment
 * @property string $highest_qualification
 */
class Auditor extends \yii\db\ActiveRecord {

    public $dob;
    public $appointment;

    const GENDER_MALE_CODE = 1;
    const GENDER_FEMALE_CODE = 2;
    const GENDER_MALE_NAME = "Male";
    const GENDER_FEMALE_NAME = "Female";
    const SCENARIO_UPLOAD = "scenario-upload";

    public $cvFile;
    public $employmentLetter;

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return 'auditor';
    }

    public function behaviors() {
        return [
            'bedezign\yii2\audit\AuditTrailBehavior'
        ];
    }

    public function scenarios() {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_UPLOAD] = ['cvFile', 'employmentLetter'];
        return $scenarios;
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            [['vote_id', 'first_name', 'last_name', 'gender', 'check_no', 'email', 'phone_number', 'professional'], 'required'],
            [['vote_id', 'gender', 'user_id'], 'integer'],
            [['email'], 'email'],
            [['cvFile', 'employmentLetter'], 'file', 'skipOnEmpty' => true, 'extensions' => 'pdf,doc, docx'],
            [['first_name', 'last_name'], 'validateName', 'on' => 'create'],
            [['first_name', 'dob', 'appointment', 'middle_name', 'last_name', 'title_position'], 'string', 'max' => 255],
            [['phone_number', 'email', 'fullname', 'check_no', 'highest_qualification'], 'string', 'max' => 200],
        ];
    }

    public function validateName($attribute, $params, $validator) {
        $exists = Auditor::find()
                ->where(['first_name' => $this->first_name, 'last_name' => $this->last_name, 'middle_name' => $this->middle_name, 'vote_id' => $this->vote_id])
                ->one();

        if ($exists) {
            $this->addError($attribute, 'Sorry! Already exists');
        }
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels() {
        return [
            'id' => 'ID',
            'vote_id' => 'Vote',
            'user_id' => 'User',
            'first_name' => 'First Name',
            'middle_name' => 'Middle Name',
            'last_name' => 'Last Name',
            'title_position' => 'Title Position',
            'phone_number' => 'Phone Number',
            'email' => 'Email',
            'fullname' => 'Full Name',
            'gender' => 'Gender',
            'data_of_birth' => 'Data Of Birth',
            'check_no' => 'Check No',
            'appointment' => 'Date Of Appointment',
            'dob' => 'Date of Birth',
            'professional' => 'Profesional Qualification',
            'employmentLetter' => 'Employment/Transfer Letter',
            'highest_qualification' => 'Highest Education Qualification',
            'cvFile' => 'CV'
        ];
    }

    public function getFullname() {
        return $this->first_name . " " . $this->middle_name . " " . $this->last_name;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getVote() {
        return $this->hasOne(Vote::className(), ['id' => 'vote_id']);
    }

    public function getUser() {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    public function getGenderName() {
        switch ($this->gender) {
            case self::GENDER_MALE_CODE:
                return self::GENDER_MALE_NAME;

                break;
            case self::GENDER_FEMALE_CODE:
                return self::GENDER_FEMALE_NAME;
                break;

            default:return "";
                break;
        }
    }

    public static function genderoptions() {
        $options = [];
        $options[] = ['id' => self::GENDER_MALE_CODE, 'name' => self::GENDER_MALE_NAME];
        $options[] = ['id' => self::GENDER_FEMALE_CODE, 'name' => self::GENDER_FEMALE_NAME];
        return $options;
    }

    public function beforeSave($insert) {
        $formatter = Yii::$app->formatter;
        if ($this->appointment)
            $this->date_of_appointment = $formatter->asTimestamp($this->appointment);
        if ($this->dob)
            $this->data_of_birth = $formatter->asTimestamp($this->dob);
        $this->professional = json_encode($this->professional);
        if ($insert) {
            //create user account
            $user = new UserForm;
            $name = $this->first_name . "." . $this->last_name . rand(0, 15);
            $user->username = $name;
            $user->password = $name;
            $user->type = \common\models\User::TYPE_IAG_OFFICER;
            $user->status = \common\models\User::STATUS_ACTIVE;
            $user->email = $this->email;
            $user->voteid = $this->vote_id;
            $model = $user->save();
            if ($model) {
                //user saved.
                $this->user_id = $model->id;
            } else {
                print_r($user->errors);
            }
        }
        return parent::beforeSave($insert);
    }

    public function upload($type) {
        $isLetter = $type == 'letter';
        $array = [];
        $path = Yii::getAlias('@frontend') . "/web/" . ($isLetter ? Yii::getAlias("@uploadLetterPath") : Yii::getAlias("@uploadCvPath"));
        if (!file_exists($path)) {
            mkdir($path, 0777, true);
        }
        $file = $isLetter ? $this->employmentLetter : $this->cvFile;
        $newName = $filename = time() . Yii::$app->security->generateRandomString();
        $fullname = $newName . '.' . $file->extension;
        $path = $path . "/$fullname";
        $fileAttribute = ['original_name' => $file->baseName,
            'full_name' => $fullname,
            'extension' => $file->extension, 'real_name' => $newName];
        $file->saveAs($path);
        if ($isLetter)
            $this->employment_letter = json_encode($fileAttribute);
        else
            $this->cv_file = json_encode($fileAttribute);

        return true;
    }

    public function afterFind() {
        $formatter = Yii::$app->formatter;
        $this->dob = $formatter->asDate($this->data_of_birth ?: NULL);
        $this->appointment = $formatter->asDate($this->date_of_appointment ?: NULL);
        $this->professional = $this->professional ?json_decode($this->professional, TRUE):[];
        return parent::afterFind();
    }

    public function getCvLink() {
        if (!$this->cv_file)
            return;
        $path = Yii::getAlias('@frontendUrl') . "/" . Yii::getAlias("@uploadCvPath");
        $file = json_decode($this->cv_file, true);
        return $path . "/" . $file["full_name"];
    }

    public function getLetterLink() {
        if (!$this->employment_letter)
            return;
        $path = Yii::getAlias('@frontendUrl') . "/" . Yii::getAlias("@uploadLetterPath");
        $file = json_decode($this->employment_letter, true);
        return $path . "/" . $file["full_name"];
    }

}
