<?php

namespace common\models;

use common\helpers\Date;
use Yii;
use backend\models\UserForm;
use common\models\User;

/**
 * This is the model class for table "CommitteeMember".
 *
 * @property integer $id
 * @property integer $vote_id
 * @property integer $user_id
 * @property string $first_name
 * @property string $middle_name
 * @property string $last_name
 * @property string $title_position
 * @property string $phone_number
 * @property string $email
 * @property integer $gender
 * @property integer $date_of_appointment
 * @property string $highest_qualification
 */
class CommitteeMember extends \yii\db\ActiveRecord {

    const GENDER_MALE_CODE = 1;
    const GENDER_FEMALE_CODE = 2;
    const GENDER_MALE_NAME = "Male";
    const GENDER_FEMALE_NAME = "Female";
    const SCENARIO_UPLOAD_APPOINTMENT_LETTER = "scenario-appointment-letter";
    public $appointmentStartDate;
    public $appointmentEndDate;
    public $appointmentFile;

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return 'committee_member';
    }

    public function behaviors() {
        return [
            'bedezign\yii2\audit\AuditTrailBehavior'
        ];
    }
    
     public function scenarios() {
        $scenarios = parent::scenarios();
        $scenarios[self::SCENARIO_UPLOAD_APPOINTMENT_LETTER] = ['approvalLetter', 'letter_reference'];
        return $scenarios;
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            [['phone_number','email','highest_qualification','vote_id', 'first_name', 'last_name', 'gender', 'appointmentStartDate', 'appointmentEndDate','member_type','professional'], 'required'],
            [['vote_id', 'gender', 'user_id', 'appointment_end_at', 'appointment_start_at'], 'integer',],
            [['email'], 'email'],
            [['appointmentFile'], 'file', 'skipOnEmpty' => false, 'extensions' => 'pdf, doc, docx',
                'on' => self::SCENARIO_UPLOAD_APPOINTMENT_LETTER],
            [['first_name', 'last_name'], 'validateName', 'on' => 'create'],
            [['first_name', 'middle_name', 'last_name', 'title_position','member_type','professional'], 'string', 'max' => 255],
            [['phone_number', 'email', 'fullname',  'highest_qualification'], 'string', 'max' => 200],
        ];
    }

    public function validateName($attribute, $params, $validator) {
        $exists = CommitteeMember::find()
                ->where(['first_name' => $this->first_name, 'last_name' => $this->last_name, 'middle_name' => $this->middle_name, 'vote_id' => $this->vote_id])
                ->one();

        if ($exists) {
            $this->addError($attribute, 'Sorry! Already exists');
        }
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels() {
        return [
            'id' => 'ID',
            'vote_id' => 'Vote',
            'user_id' => 'User',
            'first_name' => 'First Name',
            'middle_name' => 'Middle Name',
            'last_name' => 'Last Name',
            'title_position' => 'Title Position',
            'phone_number' => 'Phone Number',
            'email' => 'Email',
            'fullname' => 'Full Name',
            'gender' => 'Gender',
            'appointmentEndDate' => 'Appointment End Date',
            'appointmentStartDate' => 'Appointment Start Date',
            'date_of_appointment' => 'Date Of Appointment',
            'highest_qualification' => 'Highest Education Qualification',
            "appointmentFile"=>"Appointment Letter"
        ];
    }

    public function getFullname() {
        return $this->first_name . " " . $this->middle_name . " " . $this->last_name;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getVote() {
        return $this->hasOne(Vote::className(), ['id' => 'vote_id']);
    }

    public function getUser() {
        return $this->hasOne(User::className(), ['id' => 'user_id']);
    }

    public function getGenderName() {
        switch ($this->gender) {
            case self::GENDER_MALE_CODE:
                return self::GENDER_MALE_NAME;

                break;
            case self::GENDER_FEMALE_CODE:
                return self::GENDER_FEMALE_NAME;
                break;

            default:return "";
                break;
        }
    }

    public static function genderoptions() {
        $options = [];
        $options[] = ['id' => self::GENDER_MALE_CODE, 'name' => self::GENDER_MALE_NAME];
        $options[] = ['id' => self::GENDER_FEMALE_CODE, 'name' => self::GENDER_FEMALE_NAME];
        return $options;
    }

    public function beforeSave($insert) {
        $formatter = Yii::$app->formatter;
        $this->appointment_start_at = $formatter->asTimestamp($this->appointmentStartDate);
        $this->appointment_end_at = $formatter->asTimestamp($this->appointmentEndDate);
        return parent::beforeSave($insert);
    }

    public function afterFind() {
        $formatter = Yii::$app->formatter;
        $this->appointmentStartDate = $formatter->asDate($this->appointment_start_at ?: NULL);
        $this->appointmentEndDate = $formatter->asDate($this->appointment_end_at ?: NULL);
        return parent::afterFind();
    }

    public function uploadLetter() {
        $array = [];
        $path = Yii::getAlias('@frontend') . "/web/" . Yii::getAlias("@uploadLetterPath");
        if (!file_exists($path)) {
            mkdir($path, 0777, true);
        }
        $file = $this->appointmentFile;
        $newName = $filename = time() . Yii::$app->security->generateRandomString();
        $fullname = $newName . '.' . $file->extension;
        $path = $path . "/$fullname";
        $fileAttribute = ['original_name' => $file->baseName,
            'full_name' => $fullname,
            'extension' => $file->extension, 'real_name' => $newName];
        $file->saveAs($path);
        $this->appointment_letter_file = json_encode($fileAttribute);
        return true;
    }

    

    /**
     * 
     * @return type
     */
    public function getLetterLink() {
        if (!$this->appointment_letter_file)
            return;
        $path = Yii::getAlias('@frontendUrl'). "/" . Yii::getAlias("@uploadLetterPath");
        $file = json_decode($this->appointment_letter_file, true);
        return $path . "/" . $file["full_name"];
    }

}
