<?php

namespace common\models;


use Yii;
use common\helpers\Date;

/**
 * This is the model class for table "quarter".
 *
 * @property integer $id
 * @property integer $financial_year_id
 * @property string $code
 * @property string $name
 * @property integer $start_date
 * @property integer $end_date
 *
 * @property Implementation[] $implementations
 * @property Plan[] $plans
 * @property FinancialYear $financialYear
 * @property ReportAttachment[] $reportAttachments
 */
class Quarter extends \yii\db\ActiveRecord
{
     
    public $initial_date;
    public $final_date;
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'quarter';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            [['financial_year_id', 'code','is_current', 'initial_date', 'final_date'], 'required'],
            [['financial_year_id', 'start_date', 'end_date'], 'integer'],
            [['code'], 'string', 'max' => 45],
            [['code','financial_year_id'],'unique','targetAttribute'=>['code','financial_year_id'],'message' => 'Quarter/Year combination already exists'],
            [['name'], 'string', 'max' => 150],
            [['financial_year_id'], 'exist', 'skipOnError' => true, 'targetClass' => FinancialYear::className(), 'targetAttribute' => ['financial_year_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'financial_year_id' => 'Financial Year',
            'code' => 'Code',
            'name' => 'Quarter Name',
            'initial_date' => 'Start Date',
            'final_date' => 'End Date',
            'is_current'=>'Is Current'
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getImplementations()
    {
        return $this->hasMany(Implementation::className(), ['quarter_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getPlans()
    {
        return $this->hasMany(Plan::className(), ['quarter_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getFinancialYear()
    {
        return $this->hasOne(FinancialYear::className(), ['id' => 'financial_year_id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getReportAttachments()
    {
        return $this->hasMany(ReportAttachment::className(), ['quarter_id' => 'id']);
    }
    
    public static function choices(){
        return [
            ['value'=>'Q1','name'=>'Q1'],
            ['value'=>'Q2','name'=>'Q2'],
            ['value'=>'Q3','name'=>'Q3'],
            ['value'=>'Q4','name'=>'Q4'],
        ];
    }
     public static function listQuarters($financial_year_id){
        return  self::find()
                 ->where(['financial_year_id' => $financial_year_id])
    ->orderBy('id')
    ->all();
       
    }
    public function beforeSave($insert) {
           $this->start_date = Date::toTimestamp ($this->initial_date);
           $this->end_date = Date::toTimestamp ($this->final_date);
           Quarter::updateAll(['is_current'=>0]);
           $this->name = $this->code." - ".$this->financialYear->code;
        return parent::beforeSave($insert);
    }
    public function afterFind() {
        $this->initial_date = Date::toDateString ($this->start_date);
        $this->final_date = Date::toDateString ($this->end_date); 
        return parent::afterFind();
    }
}
