<?php

namespace common\models;

use common\commands\AddToTimelineCommand;
use common\models\query\UserQuery;
use Yii;
use yii\behaviors\AttributeBehavior;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii\web\IdentityInterface;
use common\models\UserVote;
use common\models\Vote;
use common\models\RbacAuthAssignment;
use yii\helpers\Html;

/**
 * User model
 *
 * @property integer $id
 * @property string $username
 * @property string $password_hash
 * @property string $email
 * @property string $auth_key
 * @property string $access_token
 * @property string $oauth_client
 * @property string $oauth_client_user_id
 * @property string $publicIdentity
 * @property integer $status
 * @property integer $created_at
 * @property integer $updated_at
 * @property integer $logged_at
 * @property string $password write-only password
 *
 * @property \common\models\UserProfile $userProfile
 */
class User extends ActiveRecord implements IdentityInterface {

    const STATUS_NOT_ACTIVE = 1;
    const STATUS_ACTIVE = 2;
    const PASSWORD_EXPIRE_YES = 1;
    const PASSWORD_EXPIRE_NO = 0;
    const STATUS_DELETED = 3;
    const TYPE_HQ_OFFICER = 1;
    const TYPE_IAG_OFFICER = 2;
    const TYPE_CAG_OFFICER = 3;
    const TYPE_OTHERS = 4;
    const TYPE_NAME_HQ_OFFICER = 'HQ USER';
    const TYPE_NAME_IAG_OFFICER = 'VOTE USER';
    const TYPE_NAME_CAG_OFFICER = 'CAG USER';
    const TYPE_NAME_OTHERS = 'OTHERS';
    const ROLE_USER = 'user';
    const ROLE_MANAGER = 'manager';
    const ROLE_ADMINISTRATOR = 'administrator';
    const EVENT_AFTER_SIGNUP = 'afterSignup';
    const EVENT_AFTER_LOGIN = 'afterLogin';

    public $votes = [];
    public $type_id;
    

    /**
     * @inheritdoc
     */
    public static function tableName() {
        return '{{%user}}';
    }

    /**
     * @return UserQuery
     */
    public static function find() {
        return new UserQuery(get_called_class());
    }

    /**
     * @inheritdoc
     */
    public function behaviors() {
        return [
            TimestampBehavior::className(),
            'auth_key' => [
                'class' => AttributeBehavior::className(),
                'attributes' => [
                    ActiveRecord::EVENT_BEFORE_INSERT => 'auth_key'
                ],
                'value' => Yii::$app->getSecurity()->generateRandomString()
            ],
            'access_token' => [
                'class' => AttributeBehavior::className(),
                'attributes' => [
                    ActiveRecord::EVENT_BEFORE_INSERT => 'access_token'
                ],
                'value' => function () {
                    return Yii::$app->getSecurity()->generateRandomString(40);
                }
            ],
            'bedezign\yii2\audit\AuditTrailBehavior'
        ];
    }

    /**
     * @param string $id user_id from audit_entry table
     * @return mixed|string
     */
    public static function userIdentifierCallback($id) {
        $user = self::findOne($id);
        return $user ? Html::a($user->username, ['/user/view', 'id' => $user->id]) : $id;
    }

    /**
     * @param string $identifier user_id from audit_entry table
     * @return mixed|string
     */
    public static function filterByUserIdentifierCallback($identifier) {
        return static::find()->select('id')
                        ->where(['like', 'username', $identifier])
                        ->orWhere(['like', 'email', $identifier])
                        ->orderBy(['username' => SORT_DESC])
                        ->column();
    }

    /**
     * @return array
     */
    public function scenarios() {
        return ArrayHelper::merge(
                        parent::scenarios(), [
                    'oauth_create' => [
                        'oauth_client', 'oauth_client_user_id', 'email', 'username', '!status'
                    ]
                        ]
        );
    }

    /**
     * @inheritdoc
     */
    public function rules() {
        return [
            [['username', 'email'], 'unique'],
           
            [['type'], 'integer'],
            ['status', 'default', 'value' => self::STATUS_NOT_ACTIVE],
            ['status', 'in', 'range' => array_keys(self::statuses())],
            ['type', 'in', 'range' => array_keys(self::usertypes())],
            [['username'], 'filter', 'filter' => '\yii\helpers\Html::encode'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels() {
        return [
            'username' => Yii::t('common', 'Username'),
            'email' => Yii::t('common', 'E-mail'),
            'status' => Yii::t('common', 'Status'),
            'password_expire'=>Yii::t('common', 'Password Expiry ?'),
            'type' => Yii::t('common', 'User Type'),
            'access_token' => Yii::t('common', 'API access token'),
            'created_at' => Yii::t('common', 'Created at'),
            'updated_at' => Yii::t('common', 'Updated at'),
            'logged_at' => Yii::t('common', 'Last login'),
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserProfile() {
        return $this->hasOne(UserProfile::className(), ['user_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getUserVotes() {
        return $this->hasMany(UserVote::className(), ['user_id' => 'id']);
    }

    public static function getFullname($user_id) {
       $profile = UserProfile::getProfileByUserId($user_id);     
        if ($profile){
            if($profile->firstname || $profile->lastname)
                return $profile->firstname . " " . $profile->lastname;
        }    
        $user = self::findOne(['id' =>$user_id]);
        if ($user)
            return $user->username;
        else
            return "";
    }

    /**
     * @inheritdoc
     */
    public static function findIdentity($id) {
        return static::find()
                        ->active()
                        ->andWhere(['id' => $id])
                        ->one();
    }

    /**
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null) {
        return static::find()
                        ->active()
                        ->andWhere(['access_token' => $token, 'status' => self::STATUS_ACTIVE])
                        ->one();
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username) {
        return static::find()
                        ->active()
                        ->andWhere(['username' => $username, 'status' => self::STATUS_ACTIVE])
                        ->one();
    }

    /**
     * Finds user by username or email
     *
     * @param string $login
     * @return static|null
     */
    public static function findByLogin($login) {
        return static::find()
                        ->active()
                        ->andWhere(['or', ['username' => $login], ['email' => $login]])
                        ->one();
    }

    /**
     * @inheritdoc
     */
    public function getId() {
        return $this->getPrimaryKey();
    }

    /**
     * @inheritdoc
     */
    public function getAuthKey() {
        return $this->auth_key;
    }

    /**
     * @inheritdoc
     */
    public function validateAuthKey($authKey) {
        return $this->getAuthKey() === $authKey;
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return boolean if password provided is valid for current user
     */
    public function validatePassword($password) {
        return Yii::$app->getSecurity()->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function setPassword($password) {
        $this->password_hash = Yii::$app->getSecurity()->generatePasswordHash($password);
    }

    /**
     * Returns user types list
     * @return array|mixed
     */
    public static function usertypes() {
        return [
            self::TYPE_HQ_OFFICER => Yii::t('common', self::TYPE_NAME_HQ_OFFICER),
            self::TYPE_IAG_OFFICER => Yii::t('common', self::TYPE_NAME_IAG_OFFICER),
            self::TYPE_CAG_OFFICER => Yii::t('common', self::TYPE_NAME_CAG_OFFICER),
            self::TYPE_OTHERS => Yii::t('common', self::TYPE_NAME_OTHERS)
        ];
    }

    /**
     * Returns user statuses list
     * @return array|mixed
     */
    public static function statuses() {
        return [
            self::STATUS_NOT_ACTIVE => Yii::t('common', 'Not Active'),
            self::STATUS_ACTIVE => Yii::t('common', 'Active'),
            self::STATUS_DELETED => Yii::t('common', 'Deleted')
        ];
    }
   public static function expireoptions() {
        return [
            self::PASSWORD_EXPIRE_NO => Yii::t('common', 'NO'),
            self::PASSWORD_EXPIRE_YES=> Yii::t('common', 'YES'),
            
        ];
    }
    /**
     * Creates user profile and application event
     * @param array $profileData
     */
    public function afterSignup(array $profileData = []) {
        $this->refresh();
        Yii::$app->commandBus->handle(new AddToTimelineCommand([
            'category' => 'user',
            'event' => 'signup',
            'data' => [
                'public_identity' => $this->getPublicIdentity(),
                'user_id' => $this->getId(),
                'created_at' => $this->created_at
            ]
        ]));
        $profile = new UserProfile();
        $profile->locale = Yii::$app->language;
        $profile->load($profileData, '');
        $this->link('userProfile', $profile);
        $this->trigger(self::EVENT_AFTER_SIGNUP);
        // Default role
        $auth = Yii::$app->authManager;
        $auth->assign($auth->getRole(User::ROLE_USER), $this->getId());
    }

    /**
     * @return string
     */
    public function getPublicIdentity() {
        if ($this->userProfile && $this->userProfile->getFullname()) {
            return $this->userProfile->getFullname();
        }
        if ($this->username) {
            return $this->username;
        }
        return $this->email;
    }

    public function afterFind() {

        parent::afterFind();
        $this->type_id = $this->type;
        $uservote = UserVote::findOne([
                    'user_id' => $this->id,
        ]);
        $items = $this->userVotes;
        if ($items):
            foreach ($items as $item):
                $this->votes[] = $item->vote_id;
            endforeach;
        endif;
    }

    public static function isIAOfficer() {
        $item = static::find()
                ->andWhere(['id' => Yii::$app->user->id, 'type' => self::TYPE_IAG_OFFICER])
                ->one();
        if ($item)
            return true;
        else
            return false;
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getAuditor() {

        return $this->hasOne(Auditor::className(), ['user_id' => 'id']);
    }

    public function getfirst_name() {
        return $this->auditor ? $this->auditor->first_name : $this->username;
    }

    public function getlast_name() {
        return $this->auditor ? $this->auditor->last_name : $this->username;
    }

    public function getVotes() {
        
        $uservotes = UserVote::findAll(['user_id' => $this->id]);
      $votearray = [];
       

        if ($uservotes && $this->id != 1) {
            foreach ($uservotes as $item):
                $vote = Vote::findOne($item->vote_id);
                $votearray[] = ['id' => $vote->id, 'name' => $vote->name . " - " . $vote->code];
            endforeach;
            return $votearray;
        }
        return $votearray;
    }

    public function getVotesvalue() {
        $votes = $this->getVotes();
        
        if ($votes) {
            $separator = ", ";
            $i = 0; $value = "";
            foreach ($votes as $vote){
                $i++; 
            if ($i > 1)
               $value = $value.$separator . $vote['name'];
                else
                  $value =  $vote['name'];
            }
        }else {
            $value = "( not set )";
        }
        return $value;
    }
    public function getRoles(){
         $userroles = RbacAuthAssignment::findAll(['user_id' => $this->id]);
      $array = [];
     if ($userroles) {
            foreach ($userroles as $item):
              $array[] = ['id' => $item->item_name, 'name' => $item->item_name];
            endforeach;
            
        }
        return $array;
       
    }
     public function getExpired(){
         if($this->password_expire && $this->status==self::STATUS_NOT_ACTIVE && $this->end_date<time()-\backend\models\UserForm::DAY_SECONDS)
      return true; else return false;
       
    }
    public function getRolesvalue() {
        $roles = $this->getRoles();
        
        if ($roles) {
            $separator = ", ";
            $i = 0; $value="";
            foreach ($roles as $vote){
                $i++;
            if ($i > 1)
               $value = $value.$separator . $vote['name'];
                else
                    $value =  $vote['name'];
            }
        }else {
            $value = "(not set)";
        }
        return $value;
    }
    
    
    

}
